"""

          ARIA -- Ambiguous Restraints for Iterative Assignment

                 A software for automated NOE assignment

                               Version 2.2


Copyright (C) Benjamin Bardiaux, Michael Habeck, Therese Malliavin,
              Wolfgang Rieping, and Michael Nilges

All rights reserved.


NO WARRANTY. This software package is provided 'as is' without warranty of
any kind, expressed or implied, including, but not limited to the implied
warranties of merchantability and fitness for a particular purpose or
a warranty of non-infringement.

Distribution of substantively modified versions of this module is
prohibited without the explicit permission of the copyright holders.

$Author: bardiaux $
$Revision: 1.2 $
$Date: 2006/12/18 17:27:23 $
"""



from TypeChecking import *
from threading import Thread
from aria import *
from Settings import *
from xmlutils import XMLElement, XMLBasePickler

class HostSettings(Settings):

    def create(self):

        d = {}

        d['n_cpu'] = PositiveInteger()
        d['command'] = String()
        d['executable'] = Path(exists = 0)
        d['enabled'] = YesNoChoice()

        descr = """If set to '%s' (default), we use the absolute path of the csh-script to launch the calculation. Otherwise, only its basename is used."""
        
        d['use_absolute_path'] = YesNoChoice(description = descr % str(YES))

        return d

    def create_default_values(self):

        d = {}

        d['n_cpu'] = 1
        d['command'] = 'csh'
        d['enabled'] = YES
        d['use_absolute_path'] = YES

        return d

class JobSchedulerSettings(Settings):

    def create(self):

        s = 'Default command to run a remote job, e.g. "csh", "ssh [hostname] csh" or "rsh [hostname] csh", etc. A job (csh script) is then started as follows: COMMAND job.'

        keywords = {'host_list': TypeEntity(LIST),
                    'default_command': String(description = s)}
        
        return keywords

class JobManager(AriaBaseClass):
    """
    base-class common to all job-managers
    """

    def shutdown(self):
        pass

class JobSettings(Settings):
    def create(self):
        from Settings import Path, String
        
        d = {}
        
        d['command'] = String()
        d['script'] = Path()
        d['working_directory'] = Path()
        d['use_absolute_path'] = YesNoChoice()

        return d

class Job(Thread, AriaBaseClass):
    
    cmd_template = 'cd %(working_directory)s; %(command)s %(script)s &'

    def __init__(self, settings, *args, **kw):

        check_type(settings, 'JobSettings')

        Thread.__init__(self, *args, **kw)
        AriaBaseClass.__init__(self, settings)

        self.__stop = 0
        
        ## callbacks
        self.setCallback(None)

    def stop(self):
        self.__stop = 1

    def isStopped(self):
        return self.__stop

    def setCallback(self, f):
        self.__callback = f

    def getCallback(self):
        return self.__callback

    def patch(self, host):
        """
        attempts to patch host-specific settings'
        """

        settings = self.getSettings()
        settings.update(host)
        
        d = {'executable': host['executable']}

        f = open(settings['script'])
        s = f.read() % d
        f.close()

        ## write new csh script

        f = open(settings['script'], 'w')
        f.write(s)
        f.close()

    def run(self):

        import os, time

        ## compile filename that will be polled during runtime
        
        filename = os.path.join(self.getSettings()['working_directory'],
                                'done')

        try:
            os.unlink(filename)
        except:
            pass
        
        ## start job

        d = self.getSettings().as_dict()

        ## if use shall use the local filename of the script,
        ## modify name. this gimmick is necessary for certain
        ## queuing systems.

        if d['use_absolute_path'] <> YES:
            d['script'] = os.path.basename(d['script'])

        command = self.cmd_template % d
        msg = 'Starting job: "%s"'
        self.message(msg % command)
        os.system(command)
        
        terminated = 0

        while not terminated and not self.isStopped():
            terminated = os.path.exists(filename)
            time.sleep(5.)

        if not self.isStopped():
            self.message('Job "%s" completed.' % command)

        else:
            self.debug('Job "%s" has been canceled.' % command)

        ## notify that we are done.

        if not self.isStopped():
            f = self.getCallback()

            if f is not None:
                f(self)

class JobScheduler(JobManager):
    
    def __init__(self, settings):

        check_type(settings, 'JobSchedulerSettings')
        AriaBaseClass.__init__(self, settings, name = 'Job manager')
        self.set_callback(None)

    def __check_content(self, filename, match):
        """
        reads the file 'filename'. if not equal
        to 'match' a string giving the reason is returned, None otherwise.
        """

        reason = None

        try:
            f = open(filename)
        except:
            reason = 'connection failed or unable to open temporary file.'

        if reason is None:

            s = 'dummy'

            try:
                s = f.read()[:-1]
            except:
                reason = """could not read temporary file or temporary file is empty.'"""

            if s <> match:
                reason = ''

        try:
            f.close()
        except:
            pass

        return reason

    def probeHosts(self, project):
        """
        Probes the commands specfied for every item in the host-list.
        'project' is actually an InfraStructure instance. Infrastructure,
        however, will be absorbed in Project anyway...
        """

        cmd_template = 'cd %(working_directory)s; %(command)s %(script)s' + \
                       ' %(message)s %(output_filename)s'

        import os, random

        hosts = self.getSettings()['host_list']

        temp = project.get_temp_path()
        working_dir = os.path.join(project.get_aria_path(), 'src/csh')
        src = os.path.join(working_dir, 'check_host.csh')
        
        #script = os.path.join(working_dir, 'check_host.csh')

        #d = {'working_directory': working_dir}

        passed = 1

        ## BARDIAUX 2.2
        # move src/csh/check_host.csh to tempdir/check_host.csh before running
        from tools import copy_file
        script = os.path.join(temp, 'check_host.csh')
        
        try:
            copy_file(src, script)
        except Exception, msg:
            msg = 'Could not copy %s to %s' % (src, script)            
            self.error(Exception, msg)

        d = {'working_directory': temp}
        ##
        
        for host in hosts:

            if host['enabled'] <> YES:
                continue

            d['command'] = host['command']

            tag = str(random.random())
            d['message'] = tag
            
            output_filename = os.path.join(temp, \
                                           d['command'].replace(' ', '_'))

            d['output_filename'] = output_filename

            if host['use_absolute_path'] <> YES:
                d['script'] = os.path.basename(script)
            else:
                d['script'] = script

            os.system(cmd_template % d)

            reason = self.__check_content(output_filename, tag)

            if reason is not None:
                
                ## try to read file again, after a delay of 1s

                import time
                time.sleep(2.)

                reason = self.__check_content(output_filename, tag)

            msg = 'Command "%s" ... ' % d['command']
            
            if reason is None:
                self.message(msg + 'ok.')
                
            else:

                if reason == '':
                    msg += 'failed.'
                else:
                    msg += 'failed (%s)' % reason

                self.warning(msg)

            passed = passed and reason is None

            if not passed:
                break

        return passed

    def job_done(self, job):
        host = self.running_jobs[job]
        del self.running_jobs[job]

        self.dispatch_job(host)

    def dispatch_job(self, host):

        if not len(self.jobs):
            if not len(self.running_jobs):
                self.done()

        ## start new job
            
        else:
            
            ## Get script and remove it from job-list
            job = self.jobs.pop()

            ## patch host-specific settings
            ## and set callback
            job.patch(host)
            job.setCallback(self.job_done)

            self.running_jobs[job] = host
            job.start()

    def go(self, job_list):

        check_list(job_list)

        self.jobs = job_list
        self.__done = 0

        ## compile host-list

        host_list = []

        for host in self.getSettings()['host_list']:
            if host['enabled'] == YES:
                host_list += [host] * host['n_cpu']

        ## if host-list is empty, we are done.

        if not host_list:
            s = 'Empty host-list. Ensure that at least 1 host is enabled.'
            self.message(s)
            self.done()

        ## Start as many jobs as cpus are in host-list
            
        self.running_jobs = {}
        [self.dispatch_job(host) for host in host_list]

    def done(self):
        
        if self.__callback is not None:
            self.__callback()
            
        self.__done = 1

    def is_done(self):
        return self.__done

    def set_callback(self, f):
        """
        the callback, 'f', is called when all jobs have been
        processed.
        """

        self.__callback = f

    def get_callback(self):
        return self.__callback

    def shutdown(self):
        """
        called whenever an error occurs (only if AriaBaseClass.error
        was called)
        """

        ## empty list of unfinished job
        self.jobs = []

        ## stop running jobs
        [job.stop() for job in self.running_jobs]

        self.message('shutdown.')

class HostSettingsXMLPickler(XMLBasePickler):

    order = ('enabled', 'command', 'executable', 'n_cpu', 'use_absolute_path')
    
    def _xml_state(self, x):

        e = XMLElement(tag_order = self.order)

        e.enabled = x['enabled']
        e.n_cpu = x['n_cpu']
        e.command = x['command']
        e.executable = x['executable']
        e.use_absolute_path = x['use_absolute_path']

        return e

    def load_from_element(self, e):
        s = HostSettings()
        
        s['enabled'] = str(e.enabled)
        s['n_cpu'] = int(e.n_cpu)
        s['command'] = str(e.command)
        s['executable'] = str(e.executable)

        ## TODO: remove for release version

        if hasattr(e, 'use_absolute_path'):
            value = str(e.use_absolute_path)
        else:
            value = YES
            
        s['use_absolute_path'] = value

        return s

class JobManagerXMLPickler(XMLBasePickler):

    def _xml_state(self, x):

        e = XMLElement()

        s = x.getSettings()

        e.default_command = s['default_command']

        ## If host-list is empty, do not pickle
        ## any host.

        entity = s.getEntity('host_list')

        if entity.is_initialized():
            e.host = entity.get()

        return e

    def setup_host_list(self, s):

        for host in s['host_list']:
            if host['command'] == 'default':
                host['command'] = s['default_command']

    def load_from_element(self, e):

        from tools import as_tuple

        s = JobSchedulerSettings()

        s['default_command'] = str(e.default_command)

        ## If the host-list is empty, we leave
        ## the attached entity uninitialized.

        if hasattr(e, 'host'):
            s['host_list'] = list(as_tuple(e.host))
            self.setup_host_list(s)
        else:
            s['host_list'] = []

        jm = JobScheduler(s)

        return jm

HostSettings._xml_state = HostSettingsXMLPickler()._xml_state
JobScheduler._xml_state = JobManagerXMLPickler()._xml_state
