"""

          ARIA -- Ambiguous Restraints for Iterative Assignment

                 A software for automated NOE assignment

                               Version 2.2


Copyright (C) Benjamin Bardiaux, Michael Habeck, Therese Malliavin,
              Wolfgang Rieping, and Michael Nilges

All rights reserved.


NO WARRANTY. This software package is provided 'as is' without warranty of
any kind, expressed or implied, including, but not limited to the implied
warranties of merchantability and fitness for a particular purpose or
a warranty of non-infringement.

Distribution of substantively modified versions of this module is
prohibited without the explicit permission of the copyright holders.

$Author: bardiaux $
$Revision: 1.2 $
$Date: 2006/12/18 17:27:24 $
"""



from aria import *
from Settings import Settings
from AriaXML import XMLPickler
from Settings import ChoiceEntity
from xmlutils import XMLElement, XMLBasePickler

class ReportBaseSettings(Settings):
    
    def create(self):

        from Settings import GZipChoice

        d = {}
        d['enabled'] = GZipChoice()

        return d

class CCPNSettings(ReportBaseSettings):

    def create(self):

        from Settings import YesNoChoice, ChoiceEntity

        d = ReportBaseSettings.create(self)

        del d['enabled']

        descr = \
"""
To facilitate the exchange of data between different NMR processing- and analysis packages, ARIA supports the data model standard of the Collaborative Computing Project for NMR (CCPN). ARIA exports the following items to an existing CCPN project:

- NOE restraint lists,
- Complete analysis of a structure calculation,
- Calculated structures.

Other CCPN compliant applications can then be used to read and further process ARIA generated NOE assignments etc. For more information see the CCPN webpage at http://www.ccpn.ac.uk
"""

##        d['enabled'].setDescription(descr)

        descr = 'Export the generated NOE restraint lists for all or the last iteration.'
        
        d['export_noe_restraint_list'] = ChoiceEntity(('all', 'last', 'no'), description=descr)

        descr = 'Export the coordinates of the structures generated in the last iteration and during the water refinement.'
        
        d['export_structures'] = YesNoChoice(description=descr)

        descr = 'Export NOE assignments generated by ARIA (and those that existed beforehand). The export is done for the last iteration.'

        d['export_assignments'] = YesNoChoice(description=descr)

        return d
        
    def create_default_values(self):
        
        d = ReportBaseSettings.create_default_values(self)

        d['export_noe_restraint_list'] = 'last'
        d['export_structures'] = YES
        d['export_assignments'] = YES
        
        return d

class MolMolSettings(ReportBaseSettings):
    
    def create(self):
        d = ReportBaseSettings.create(self)
        descr = \
"""
MOLMOL .lol and .upl files will be written.
"""

        d['enabled'].setDescription(descr)
        return d

    def create_default_values(self):
        d = ReportBaseSettings.create_default_values(self)
        d['enabled'] = YES

        return d

class NOEListSettings(Settings):
    
    def create(self):

        from Settings import GZipChoice

        d = {}

        ## text list

        descr = \
"""
If enabled, ARIA creates iteration-wise text report files for all restraints stemming from NOE measurements:

1. noe_restraints.(un)ambig:
Lists all (un)ambiguous restraints. The table also contains a brief statistical analysis of the respective structure ensemble. The lists are sorted with respect to spectrum name and crosspeak number.

2. noe_restraints.assignments:
Lists all assignment possibilities (contributions) for every restraint together with some statistical quantities calculated from the respective structure ensemble.

3. noe_restraints.merged:
Reports all restraints that have been removed from the restraint-list due to merging.

4. noe_restraints.violations:
All violated restraints are reported. Restraints are sorted with respect to their upper bound violation.

5. report
Summarizes all results for the respective iteration.
"""
        
        d['text_output'] = GZipChoice(description = descr)

        ## XML list

        descr = \
"""
If enabled, ARIA creates a text-file, 'noe_restraints.xml' (for every iteration). It contains the full restraint-list represented in the portable format XML. The XML restraint-list is a complete description of ARIAs internal data structure for NOE-based restraints and thus provides all data for post-analyses of a structure calculation (cf. Python Pickles). ARIAs Python API provides functions to read, write and analyse those data.
"""
        d['xml_output'] = GZipChoice(description = descr)

        ## python pickles

        descr = \
"""
If enabled, ARIA stores the full restraint-list as Python pickle, 'noe_restraints.pickle'. This is done after an iteration has been completed. Python pickles are machine independent and easy to use when performing post-analyses of a structure calculation within Python. ARIAs Python API provides functions to read, write and analyse those data.
"""
        
        d['pickle_output'] = GZipChoice(description = descr)

        return d
    
### BARDIAUX
class UpSpecSettings(Settings):
    
    def create(self):

        from Settings import YesNoChoice, ChoiceEntity

        d = {}

        descr = \
"""
If enabled, all spectra assigned by ARIA are exported to ARIA XML format. File naming convention: [spectrum_name].assigned.xml where spectrum_name denotes the name of your spectrum (as e.g. specified in the conversion.xml file).
"""
        
        d['write_assigned'] = YesNoChoice(description = descr)


        descr = \
"""
If enabled, assignments that have not been altered by ARIA are also exported.
"""
        
        d['write_assigned_force'] = YesNoChoice(description = descr)

        descr = \
'''
Determines the iterations for which assigned spectra are exported to ARIA XML. If set to "all", assigned spectra are written after every iteration, if set to "last" only for the last iteration.
'''

        d['iteration'] = ChoiceEntity(('all', 'last'), description=descr)

        # BARDIAUX 2.2        
        descr = \
'''
Determines which peaks must be written. If set to "YES", only peaks with an unambiguous assignment are exported. Otherwise, all the peaks assigned by ARIA will be exported (i.e. peaks with ambiguous or unambiguous assignments).
'''

        d['write_unambiguous_only'] = YesNoChoice(description=descr)
        
        return d

    def create_default_values(self):

        d = {}
        d['write_assigned'] = NO
        d['write_assigned_force'] = NO
        d['iteration'] = 'last'
        d['write_unambiguous_only'] = YES

        return d

class ReportSettings(Settings):

    def create(self):

        from Settings import TypeEntity

        d = {}

        d['noe_restraint_list'] = TypeEntity('NOEListSettings')
        d['ccpn'] = TypeEntity('CCPNSettings')
        d['molmol'] = TypeEntity('MolMolSettings')
        d['spectra'] = TypeEntity('UpSpecSettings')

        
        return d

    def create_default_values(self):

        s = {'noe_restraint_list': NOEListSettings,
             'ccpn': CCPNSettings,
             'molmol': MolMolSettings,
             'spectra' : UpSpecSettings}

        d = {}

        for entity, constructor in s.items():
            s = constructor()
            s.reset()
            d[entity] = s

        return d

class ReportBaseXMLPickler(XMLBasePickler):

    order = ('enabled',)

    def _xml_state(self, x):
        e = XMLElement()
        e.enabled = x['enabled']

        return e

    def create(self):
        return ReportBaseSettings()

    def load_from_element(self, e):
        s = self.create()
        s['enabled'] = str(e.enabled)

        return s

class CCPNSettingsXMLPickler(XMLBasePickler):

    order = ('export_assignments', 'export_noe_restraint_list', 'export_structures')

    def create(self):
        return CCPNSettings()

    def _xml_state(self, x):

        e = XMLElement(tag_order = self.order)

        e.export_noe_restraint_list = x['export_noe_restraint_list']
        e.export_structures = x['export_structures']
        e.export_assignments = x['export_assignments']

        return e

    def load_from_element(self, e):

        s = self.create()
        s.reset()

        for key in s.keys():
            
            if hasattr(e, key):
                s[key] = str(getattr(e, key))

        return s

class MolMolSettingsXMLPickler(ReportBaseXMLPickler):
    def create(self):
        return MolMolSettings()
    
## BARDAIUX
class UpSpecSettingsXMLPickler(ReportBaseXMLPickler):
    
    order = ('write_assigned', 'write_assigned_force', 'iteration', 'write_unambiguous_only')

    def _xml_state(self, x):

        e = XMLElement(tag_order = self.order)
        e.write_assigned = x['write_assigned']
        e.write_assigned_force = x['write_assigned_force']
        e.iteration = x['iteration']
        e.write_unambiguous_only = x['write_unambiguous_only']

        return e

    def load_from_element(self, e):

        s = UpSpecSettings()

        s['write_assigned'] = str(e.write_assigned)
        s['write_assigned_force'] = str(e.write_assigned_force)

        if hasattr(e, 'iteration'):
            s['iteration'] = str(e.iteration)
        else:
            s['iteration'] = 'last'

        if hasattr(e, 'write_unambiguous_only'):
            s['write_unambiguous_only'] = str(e.write_unambiguous_only)
        else:
            s['write_unambiguous_only'] = YES        
            
        return s

class NOEListSettingsXMLPickler(XMLBasePickler):

    order = ('pickle_output', 'text_output', 'xml_output')

    def _xml_state(self, x):

        e = XMLElement(tag_order = self.order)
        e.pickle_output = x['pickle_output']
        e.text_output = x['text_output']
        e.xml_output = x['xml_output']

        return e

    def load_from_element(self, e):

        s = NOEListSettings()

        s['pickle_output'] = str(e.pickle_output)
        s['text_output'] = str(e.text_output)
        s['xml_output'] = str(e.xml_output)

        ## TODO: remove for release version

        if hasattr(e, 'ccpn_output'):
            s._ccpn_output = str(e.ccpn_output)
            
        return s

class ReportXMLPickler(XMLBasePickler):

    order = ('ccpn', 'molmol', 'noe_restraint_list', 'spectra')

    def _xml_state(self, x):

        e = XMLElement(tag_order = self.order)

        e.ccpn = x['ccpn']
        e.molmol = x['molmol']
        e.noe_restraint_list = x['noe_restraint_list']
        e.spectra= x['spectra']

        return e

    def load_from_element(self, e):

        s = ReportSettings()

        s['noe_restraint_list'] = e.noe_restraint_list

        ## TODO: to support project files from early
        ## alpha-versions. remove the additional code
        ## for release version.

        if hasattr(e, 'ccpn'):
            s['ccpn'] = e.ccpn
        else:

            z = CCPNSettings()
            z.reset()
##             z['enabled'] = str(e.noe_restraint_list._ccpn_output)

            del e.noe_restraint_list._ccpn_output

            s['ccpn'] = z

        if hasattr(e, 'molmol'):
            s['molmol'] = e.molmol
        else:
            z = MolMolSettings()
            z.reset()

            s['molmol'] = z

        if hasattr(e, 'spectra'):
            s['spectra'] = e.spectra
            
        else:
            z = UpSpecSettings()
            z.reset()

            s['spectra'] = z

        return s

ReportBaseSettings._xml_state = ReportBaseXMLPickler()._xml_state
CCPNSettings._xml_state = CCPNSettingsXMLPickler()._xml_state
MolMolSettings._xml_state = MolMolSettingsXMLPickler()._xml_state
NOEListSettings._xml_state = NOEListSettingsXMLPickler()._xml_state
UpSpecSettings._xml_state = UpSpecSettingsXMLPickler()._xml_state
ReportSettings._xml_state = ReportXMLPickler()._xml_state
