import os, sys

class Aria2Ccpn:

    aria_dim_codes = ['Proton1', 'Proton2', 'Hetero1', 'Hetero2']

    def __init__(self, aria_project, aria_peak_list):

        ## Here set link to Aria in memory stuff

        self.aria_project = aria_project
        self.getCcpnProject()

        if not self.ccpn_project: return

        self.setCcpnMolSystem()
        self.setupCcpnConstraintPackage()
        self.setCcpnResonancesAndShifts()
        self.setCcpnConstraintList(aria_peak_list)

        for experiment in aria_project.experiments:
            peak_list = self.createCcpnPeakList(experiment)
            self.setCcpnPeakListAssignments(peak_list, aria_peak_list)

    def getCcpnProject(self):

        from memops.api import Implementation
        from memops.general.Io import loadXmlProjectFile
        from ccp.general.Io import loadChemElementRefData
        import os
  
        self.isNewCcpnProject = 0
        self.ccpn_project = None

        ccpn_file = self.aria_project.ccpn_project_filename

        if hasattr(self.aria_project, 'ccpn_project') and \
           self.aria_project.ccpn_project:

            ## Ccpn project in memory
            
            print "Accessing CCPN project %s in memory" \
                  % (self.aria_project.ccpn_project.name)
          
            self.ccpn_project = self.aria_project.ccpn_project # or something
            
        elif ccpn_file:
            
            if os.path.exists(ccpn_file):

                ## Ccpn project not in memory but exists as xml file
                
                print "Loading CCPN project %s" % ccpn_file
                self.ccpn_project = loadXmlProjectFile(file = ccpn_file)

            else:

                ## TODO: check location etc.

                ## No ccpn project but new one should be created

                print "Creating CCPN project %s" % ccpn_file
                self.isNewCcpnProject = 1
                project_name = os.path.basename(ccpn_file).split('.')[0]
                self.ccpn_project = Implementation.Project(name = project_name)

                ## Set the location...

                path, file = os.path.split(ccpn_file)
                self.ccpn_project.path = file
                self.ccpn_project.url.path = path

                out_path = os.path.join(path, project_name)

                try:
                    os.stat(out_path)
                except:
                    os.mkdir(out_path)

                for url in self.ccpn_project.urls:
                    if url != self.ccpn_project.url:
                        url.path = out_path

                loadChemElementRefData(self.ccpn_project)

        else:
            print "No CCPN project will be created..."

    def setCcpnMolSystem(self):

        from ccpnmr.format.general.Constants import ccpNmr_kw, isAriaInput_kw
        from ccpnmr.format.converters.AriaFormat import AriaFormat
        from ccp.format.aria.sequenceIO import AriaSequenceFile, \
             AriaSequence, AriaSequenceElement
        from ccp.api import MolSystem

        aria_chains = self.aria_project.getMolecule().get_chains()

        self.ccpn_mol_system = None
        self.aria2CcpnChainDict = {}

        for mol_system in self.ccpn_project.molSystems:
            data = mol_system.findFirstApplicationData(application = ccpNmr_kw,
                                                       keyword = isAriaInput_kw)
            if data and data.value:
                self.ccpn_mol_system = mol_system

        if not self.ccpn_mol_system:

            format = AriaFormat(self.ccpn_project, verbose = 0)

            format.sequenceFile = AriaSequenceFile(name = 'none')
            ## set something so it won't read stuff!
            format.file = 'test' 

            for chain in aria_chains:

                format.sequenceFile.sequences.append(\
                    AriaSequence(chain.getSegid()))

                for residue in chain.getResidues():
                    residue_type = residue.getType()
                    if residue_type:
                        seq_code = residue.getNumber()
                        format.sequenceFile.sequences[-1].elements.append(\
                            AriaSequenceElement(seq_code, residue_type))
                    else:
                        print "Error: residue type is None"

            self.ccpn_mol_system = MolSystem.MolSystem(self.ccpn_project,
                                                     code = 'test')

            ccpn_chains = format.readSequence(molSystem = self.ccpn_mol_system,
                                              minimalPrompts = 1)

            for i in range(len(aria_chains)):
                self.aria2CcpnChainDict[aria_chains[i]] = ccpn_chains[i]

        else:

            for chain in aria_chains:
                s = chain.getSegid().strip()
                for ccpn_chain in mol_system.chains:
                    if ccpn_chain.code.strip() == s:
                        self.aria2CcpnChainDict[ariaChain] = ccpnChain
                        break
                    else:
                        m = 'could not find matching chain for segid "%s"' \
                            % ariaChain.getSegid()
                        raise Exception(m)
      
    def setupCcpnConstraintPackage(self):

        from ccp.api import Nmr
        from ccpnmr.format.general.Io import createContentStorage
  
        self.ccpn_struct_generation = Nmr.StructureGeneration(\
            self.ccpn_project)

        content_storage, constraint_set = createContentStorage(\
            'ccp.NmrConstraints', self.ccpn_project, 0, numberSet = 1)

        self.constraint_head = Nmr.NmrConstraintHead(\
            self.ccpn_project, contentStorage = content_storage)
        
        self.ccpn_struct_generation.nmrConstraintHead = self.constraint_head

    def setCcpnResonancesAndShifts(self):
  
        from ShiftAssignment import ASSIGNMENT_METHOD_STEREO_SPECIFIC
        from ShiftAssignment import ASSIGNMENT_METHOD_EQUIVALENT
        from ShiftAssignment import ASSIGNMENT_METHOD_FLOATING

        from ccp.api import Nmr, NmrConstraints
        from ccpnmr.format.general.Util import getNmrIsotope
        from memops.api import Implementation
        from memops.universal.Constants import False, True
        from ccpnmr.format.general.Constants import ccpNmr_kw
        
        self.atom2resonance = {}
        self.atom2fixedresonance = {}

        atom_groups = {}

        for experiment in self.aria_project.experiments:

            aria_shifts = experiment.getShiftList()
            ccpn_shifts = None

            if not self.isNewCcpnProject:

                ccpn_experiment = self.ccpn_project.findFirstNmrExperiment(\
                    name = experiment.getName())

                if ccpn_experiment and ccpn_experiment.shiftList:
                    ccpn_shifts = ccpn_experiment.shiftList

            if ccpn_shifts:
                isNewShiftList = False

            else:
                ccpn_shifts = Nmr.ShiftList(self.ccpn_project, \
                                            name = experiment.getName())
                isNewShiftList = True

            for assignment in aria_shifts.getShiftAssignments():

                method = assignment.getMethod()

                if method == ASSIGNMENT_METHOD_STEREO_SPECIFIC:

                    groups = [assignment.getSpinSystems()[0].getAtoms()]
                    shifts = assignment.getSpinSystems()[0].getChemicalShifts()        

                elif method == ASSIGNMENT_METHOD_EQUIVALENT:

                    groups = [assignment.getSpinSystems()[0].getAtoms()]
                    shifts = assignment.getSpinSystems()[0].getChemicalShifts()    

                elif method == ASSIGNMENT_METHOD_FLOATING:

                    groups = []
                    shifts = []

                    for spin_system in assignment.getSpinSystems():

                        groups.append(spin_system.getAtoms())
                        shifts.extend(spin_system.getChemicalShifts())

                ## keep track of info in dicts

                for i in range(len(groups)):

                    group = list(groups[i])
                    group.sort()
                    key = str(group)

                    if not atom_groups.has_key(key):
                        atom_groups[key] = {}
                        atom_groups[key]['atomGroup'] = group

                    if not atom_groups[key].has_key('shiftLists'):
                        atom_groups[key]['shiftLists'] = {}

                    if i < len(shifts):
                        atom_groups[key]['shiftLists'][ccpn_shifts] = shifts[i]
                    else:
                        atom_groups[key]['shiftLists'][ccpn_shifts] = shifts[-1]
                    if len(groups) > 1:

                        if not atom_groups[key].has_key('otherAtomGroups'):
                            atom_groups[key]['otherAtomGroups'] = []

                        for j in range(len(groups)):

                            if i <> j:

                                other_group = list(groups[j])
                                other_group.sort()

                                if other_group not in \
                                       atom_groups[key]['otherAtomGroups']:
                                    atom_groups[key]['otherAtomGroups'].append(other_group)

        ## now set the atoms... in whatever order

        done = []

        for key in atom_groups.keys():

            group_info = atom_groups[key]
            atom_group = group_info['atomGroup']
            group_infos = [group_info]

            if key in done: continue

            residue = atom_group[0].getResidue()
            chain = residue.getChain()
            
            ccpn_chain = self.aria2CcpnChainDict[chain]
            ccpn_residue = ccpn_chain.findFirstResidue(\
                seqId = residue.getNumber())

            if group_info.has_key('otherAtomGroups') and \
               group_info['otherAtomGroups']:

                for other_group in group_info['otherAtomGroups']:
                    done.append(str(other_group))
                    group_infos.append(atom_groups[str(other_group)])

            ## now do each relevant group

            ccpn_atomsets = []
            ccpn_fixed_atomsets = []
            used_resonances = []
            created_resonances = []

            for group_info in group_infos:

                group = group_info['atomGroup']

                ccpn_group = []

                for atom in group:

                    iupac_name = atom.getName()
                    ccpn_atom = ccpn_residue.findFirstAtom(name = iupac_name)

                    if ccpn_atom:
                        ccpn_group.append(ccpn_atom)
                    else:
                        m = "Error: no match for atom name %s.%s.%s in ccpn."
                        print m % (ccpn_chain.code, ccpn_residue.seqCode,
                                   iupac_name)

                if len(ccpn_group) > 1:
                    atomset_name = ccpn_group[0].chemAtom.chemAtomSet.name
                else:
                    atomset_name = ccpn_group[0].name

                ccpn_group.sort()

                ccpn_ref_atom = ccpn_group[0]
                ccpn_atomset = ccpn_ref_atom.atomSet

                if ccpn_atomset:
                    
                    check_ccpn_atoms = list(ccpn_atomset.atoms)
                    check_ccpn_atoms.sort()

                    if check_ccpn_atoms != ccpn_group:
                        raise Exception('atom sets do not match for %s' \
                                        % ccpn_atomset.name)
                    else:
                        for resonance_set in ccpn_atomset.resonanceSets:
                            for resonance in resonance_set.resonances:
                                used_resonances.append(resonance)

                else:
                    if not isNewShiftList:
                        print 'Warning: creating new atomSet %s' % atomset_name

                    ccpn_atomset = Nmr.AtomSet(self.ccpn_project,
                                               name = atomset_name,
                                               atoms = ccpn_group)

                ## always create a new FixedAtomSet based on this info!

                ccpn_fixed_atomset = NmrConstraints.FixedAtomSet(\
                    self.constraint_head, name = ccpn_atomset.name,
                    atoms = ccpn_atomset.atoms)

                ## keep track...

                ccpn_atomsets.append(ccpn_atomset)
                ccpn_fixed_atomsets.append(ccpn_fixed_atomset)

                ## check if shift is always none for this atomGroup

                for shift_list in group_info['shiftLists']:

                    if group_info['shiftLists'][shift_list].getValue() != None:
                        created_resonances.append((group_info, atomset_name))
                        break

            ## Make resonances and fixed resonances, set shifts

            isotope_code = getNmrIsotope(\
                self.ccpn_project,
                ccpn_atomsets[0].atoms[0].chemAtom.elementSymbol)

            ccpn_resonances = []
            ccpn_fixed_resonances = []

            for group_info, atomset_name in created_resonances:

                if not used_resonances:

                    name = "%s.%s" % (str(ccpn_residue.seqCode),
                                      str(atomset_name))

                    resonance = Nmr.Resonance(\
                        self.ccpn_project, name = name,
                        isotopeCode = isotope_code)
                    
                    ccpn_resonances.append(resonance)
                    
                    Implementation.AppDataString(\
                        resonance, application = ccpNmr_kw,
                        keyword = 'mappingSingleMatch', value = atomset_name)

                else:
                    ccpn_resonances = used_resonances

                ## set fixed resonances
                    
                for resonance in ccpn_resonances:

                    fixed_resonance = NmrConstraints.FixedResonance(\
                        self.constraint_head, name = resonance.name,
                        details = resonance.details,
                        isotopeCode = resonance.isotopeCode)

                    ccpn_fixed_resonances.append(fixed_resonance)
                    
                    Implementation.AppDataString(\
                        fixed_resonance, application = ccpNmr_kw,
                        keyword = 'mappingSingleMatch', value = atomset_name)

                for atom in group_info['atomGroup']:
                    
                    if not self.atom2resonance.has_key(atom):
                        self.atom2resonance[atom] = ccpn_resonances[-1]

                    if not self.atom2fixedresonance.has_key(atom):
                        self.atom2fixedresonance[atom] = \
                                                ccpn_fixed_resonances[-1]

                if isNewShiftList:

                    for shift_list in group_info['shiftLists']:

                        shift, error = group_info['shiftLists'][shift_list]
                        message = "Shift %.3f already set to %.3f for %s" 
                        
                        if shift is not None:

                            if not used_resonances:
                                Nmr.Shift(shift_list,
                                          resonance = ccpn_resonances[-1],
                                          value = shift, error = error)

                            else:
                                for resonance in used_resonances:

                                    try:
                                        Nmr.Shift(shift_list,
                                                  resonance = resonance,
                                                  value = shift, error = error)
                                    except:

                                        ccpn_shift = resonance.findFirstShift(\
                                            parentList = shift_list)
                                        
                                        a = (shift, ccpn_shift.value,
                                             group_info['atomGroup'])
                                        print message % a
                                                         
            if created_resonances and not used_resonances:

                Nmr.ResonanceSet(self.ccpn_project,
                                 resonances = ccpn_resonances,
                                 atomSets = ccpn_atomsets)

            NmrConstraints.FixedResonanceSet(self.constraint_head,
                                             resonances=ccpn_fixed_resonances,
                                             atomSets=ccpn_fixed_atomsets)

    def setCcpnConstraintList(self, aria_peak_list):

        from tools import sortProtonDimensions
        from ccp.api import NmrConstraints

        ## TODO: set name
        ccpn_restraints = NmrConstraints.DistanceConstraintList(\
            self.constraint_head, name = 'ariaTest')

        for aria_peak in aria_peak_list:

            ## TODO: Here also create a link to the peak!!!

            ccpn_restraint = NmrConstraints.DistanceConstraint(\
                ccpn_restraints)
            ccpn_restraint.upperLimit = aria_peak.getUpperBound()
            ccpn_restraint.lowerLimit = aria_peak.getLowerBound()
            ccpn_restraint.targetValue = aria_peak.getDistance()
            ccpn_restraint.weight = aria_peak.getWeight()
            ccpn_temp_contribs = []

            for aria_contrib in aria_peak.getContributions():

                for spin_pair in aria_contrib.getSpinPairs():

                    ccpn_temp_members = []

                    aria_atoms = sortProtonDimensions(\
                        spin_pair, aria_peak.getReferencePeak())          

                    if not aria_atoms:
                        print "ERROR: no aria atoms available"
                        continue

                    for aria_atom in aria_atoms:

                        if self.atom2fixedresonance.has_key(aria_atom):
                            ccpn_temp_members.append(\
                                self.atom2fixedresonance[aria_atom])
                        else:
                            print "WARNING: not recognized %s" \
                                  % str(aria_atom)

                    if ccpn_temp_members not in ccpn_temp_contribs:
                        ccpn_temp_contribs.append(ccpn_temp_members)

            for resonances in ccpn_temp_contribs:

                resonances.sort()

                if resonances[0] == resonances[1]: continue

                if not ccpn_restraint.findFirstItem(resonances = resonances):
                    NmrConstraints.DistanceConstraintItem(\
                        ccpn_restraint, resonances = resonances)

    def createCcpnPeakList(self, aria_experiment):
  
        from ccpnmr.format.general.Constants import ccpNmr_kw, isAriaInput_kw,\
             isAriaOutput_kw
        from ccpnmr.format.general.Util import createExperiment, \
             createPpmFreqDataSource
        from memops.universal.Constants import True
        from memops.api import Implementation
        from ccp.api import Nmr

        ccpn_peak_list_name = ccpn_exp_name = aria_experiment.getName()
        ccpn_peak_list = None
        ccpn_exp = None
        ccpn_data_source = None

        if not self.isNewCcpnProject:
            
            for nmr_exp in self.ccpn_project.nmrExperiments:

                if nmr_exp.name == ccpn_exp_name:
                    ccpn_exp = nmr_exp

                for source in nmr_exp.dataSources:

                    peak_lists = source.findAllPeakLists(\
                        name = ccpn_peak_list_name)

                    for peak_list in peak_lists:

                        output = peak_list.findFirstApplicationData(\
                            application = ccpNmr_kw, keyword = isAriaOutput_kw)

                        input = peak_list.findFirstApplicationData(\
                            application = ccpNmr_kw, keyword = isAriaInput_kw)

                        if output and output.value and not input:
                          ccpn_peak_list = peak_list
                          ccpn_data_source = source
                          break

                    if not ccpn_peak_list:
                        for peak_list in peak_lists:

                            input = peak_list.findFirstApplicationData(\
                                application = ccpNmr_kw,
                                keyword = isAriaInput_kw)

                            if input and input.value:
                                ccpn_data_source = source
                                break

        ## prepare mappings...

        self.aria2ccpn_peak_list = {}
        self.aria2ccpn_data_dim_ref = {}
        self.ccpn2aria_dim_code = {}

        aria_spectrum = aria_experiment.getSpectrum()
        aria_peaks = aria_spectrum.getPeaks()

        ## make the ccpn NMR experiment if it doesn't exist

        if not ccpn_exp:

            ## make a ccpn peak list from scratch: need to define experiment
            ## type, dims, ..., then create peaks, ... using the spinsystems?!

            num_dim = 0
            dimensions = []
            valid_dim_codes = []

            for dim_code in self.aria_dim_codes:

                s = getattr(aria_peaks[0], 'get%sChemicalShift' % dim_code)
                aria_shift = s()[0]

                if aria_shift:
                    
                    num_dim += 1
                    type_found = 0

                    for aria_peak in aria_peaks:

                        assignments = getattr(\
                            aria_peak, 'get%sAssignments' % dim_code)()

                        for assignment in assignments:

                          atom = assignment.getAtoms()[0]
                          dimensions.append(atom.getType())
                          self.aria2ccpn_data_dim_ref[dim_code] = None
                          valid_dim_codes.append(dim_code)
                          type_found = 1
                          break

                        if type_found:
                            break

            exp_type = None

            if dimensions[:2] == ['H','H']:

                if num_dim == 2:
                  exp_type = 'noesy.hh'

                elif num_dim == 3:

                    if dimensions[2] == 'C':
                        exp_type = 'noesy_hsqc_HCH.hhc'

                    elif dimensions[2] == 'N':
                        exp_type = 'noesy_hsqc_HNH.hhn'

                elif num_dim == 4:

                  if dimensions[2:] == ['C','N']:
                    exp_type = 'noesy_hsqc_HCNH.hhcn'

                  elif dimensions[2:] == ['N','C']:
                    exp_type = 'noesy_hsqc_HNCH.hhnc'

                  elif dimensions[2:] == ['N','N']:
                    exp_type = 'noesy_hsqc_HNNH.hhnn'

                  elif dimensions[2:] == ['C','C']:
                    exp_type = 'noesy_hsqc_HCCH.hhcc'


            if exp_type:
                ccpn_exp = createExperiment(\
                    self.ccpn_project, ccpn_exp_name, exp_type)

            else:
                m = "Unrecognized experiment type, dimensions %d, nuclei %s"
                raise Exception(m % (num_dim, dimensions))


        ## create data source if doesn't exist

        if not ccpn_data_source:

            ccpn_data_source = createPpmFreqDataSource(\
                ccpn_exp, 'ARIA', 'processed', ccpn_exp.numDim)

            ## set up dimension mapping

            for i in range(len(ccpn_data_source.dataDims)):
                
                data_dim = ccpn_data_source.dataDims[i]
                dim_code = valid_dim_codes[i]
                self.aria2ccpn_data_dim_ref[dim_code] = data_dim.dataDimRefs[0]
                self.ccpn2aria_dim_code[data_dim.dataDimRefs[0]] = dim_code

        else:

            ## use application data to get right mapping between
            ## freqDataDims and ariaDimCodes

            for data_dim in ccpn_data_source.dataDims:
                
                for dim_ref in data_dim.dataDimRefs:

                  data = dim_ref.findFirstApplicationData(\
                      application = ccpNmr_kw, keyword = 'ariaDimMapping')

                  if data and data.value:
                      self.aria2ccpn_data_dim_ref[data.value] = dim_ref
                      self.ccpn2aria_dim_code[dim_ref] = data.value

        ## make the ccpn peak list if it doesn't exist

        if not ccpn_peak_list:

            ccpn_peak_list = Nmr.PeakList(\
                ccpn_data_source, name = ccpn_peak_list_name)

            Implementation.AppDataBoolean(\
                ccpn_peak_list, application = ccpNmr_kw,
                keyword = isAriaOutput_kw, value = True)

        self.aria2ccpn_peak_list[aria_spectrum.getName()] = ccpn_peak_list

        return ccpn_peak_list

    def setCcpnPeakListAssignments(self, ccpn_peak_list, aria_restraints):
    
        from tools import sortProtonDimensions
        from ccp.api import Nmr

        ## use the restraint list to set the assignments in the peak list 

        data_dim_refs = [data_dim.dataDimRefs[0] for data_dim \
                         in ccpn_peak_list.dataSource.dataDims]

        for aria_restraint in aria_restraints:

            if aria_restraint.isMerged():
                aria_ref_restraint = aria_restraint.getEquivalentPeak()
            else:
                aria_ref_restraint = aria_restraint

            aria_peak = aria_restraint.getReferencePeak()
            aria_spectrum_name = aria_peak.getSpectrum().getName()

            if not self.aria2ccpn_peak_list.has_key(aria_spectrum_name):
                continue

            ccpn_peak_list = self.aria2ccpn_peak_list[aria_spectrum_name]

            ## TODO: make sure this is set in Ccpn2Aria!!
            ccpn_peak_serial = aria_peak.getNumber()
            ccpn_peak = ccpn_peak_list.findFirstPeak(serial = ccpn_peak_serial)

            if not ccpn_peak:
                
                ccpn_peak = Nmr.Peak(ccpn_peak_list)

                for dim_code in self.aria_dim_codes:

                    if dim_code in self.aria2ccpn_data_dim_ref:
                        data_dim_ref = self.aria2ccpn_data_dim_ref[dim_code]
                        if data_dim_ref:
                          value, error = getattr(\
                              aria_peak, 'get%sChemicalShift' % dim_code)()
                          ccpn_peak_dim = ccpn_peak.findFirstPeakDim(\
                              dim = data_dim_ref.dataDim.dim)
                          ccpn_peak_dim.value = value
                          ccpn_peak_dim.error = error
                          ccpn_peak_dim.dataDimRef = data_dim_ref

            else:        

                ## delete existing contribs: make sure right peak
                ## list selected!!

                [c.delete() for c in ccpn_peak.peakContribs]
                [c.delete() for d in ccpn_peak.peakDims
                 for c in d.peakDimContribs]

            ccpn_temp_contribs = []

            aria_assigned_atoms = {}
            
            for aria_dim_code in self.aria_dim_codes[2:]:

                aria_assigned_atoms[aria_dim_code] = []
                assignments = getattr(\
                    aria_peak, 'get%sAssignments' % aria_dim_code)()

                for assignment in assignments:
                    for atom in assignment.getAtoms():
                        if atom not in aria_assigned_atoms[aria_dim_code]:
                            aria_assigned_atoms[aria_dim_code].append(atom)

            for contrib in aria_ref_restraint.getContributions():

                for spin_pair in contrib.getSpinPairs():

                    ccpn_temp_members = []

                    aria_atoms = sortProtonDimensions(\
                        spin_pair, aria_ref_restraint.getReferencePeak())

                    [ccpn_temp_members.append(self.atom2resonance[a])
                     for a in aria_atoms]
                        
                    contrib_exists = 0
                    
                    for ccpn_temp_contrib in ccpn_temp_contribs:
                        if ccpn_temp_members == ccpn_temp_contrib[:2]:
                            contrib_exists = 1
                            break

                    if not contrib_exists:
                        ccpn_temp_members.extend([None, None])
                        ccpn_temp_contribs.append(ccpn_temp_members)

                        for atom in aria_atoms:

                            hetero = atom.getHeteroAtom()
                            
                            for i in range(2, 4):

                                dim_code = self.aria_dim_codes[i]

                                if hetero in aria_assigned_atoms[dim_code]:

                                    if hetero in self.atom2resonance:
                                        resonance = self.atom2resonance[hetero]
                                    else:
                                        print "ERROR: no ccpn hetero atom %s" \
                                              % hetero.getName()
                                        resonance = None
                                        
                                    ccpn_temp_members[i] = resonance


            for resonances in ccpn_temp_contribs:
                
                peak_contrib = Nmr.PeakContrib(ccpn_peak)

                for i in range(len(resonances)):

                    resonance = resonances[i]

                    if resonance is None: continue
                    
                    dim_code = self.aria_dim_codes[i]
                    dim_ref = self.aria2ccpn_data_dim_ref[dim_code]
                    peak_dim = ccpn_peak.findFirstPeakDim(dataDimRef = dim_ref)
                    dim_contrib = peak_dim.findFirstPeakDimContrib(\
                        resonance = resonance)

                    if dim_contrib:
                        dim_contrib.addPeakContrib(peak_contrib)
                    else:
                        Nmr.PeakDimContrib(\
                            peak_dim, resonance = resonance,
                            peakContribs = [peak_contrib])


if __name__ == '__main__':

    from AriaXML import AriaXMLPickler

    project_xml = '~/wim/AriaTest/new_run3.xml'

    aria_pickler = AriaXMLPickler()
    aria_project = aria_pickler.load(project_xml)
    aria_project.read_molecule()
    aria_project.read_data()

    aria_project.ccpn_project_filename = '/tmp/test.xml'

    n_iterations = len(aria_project.getProtocol().\
                       getSettings()['iteration_settings']) - 1

    it_path = aria_project.getInfrastructure().get_iteration_path(n_iterations)

    import AriaPeak

    AriaPeak.AriaPeakXMLPickler.spectra = {}
    for exp in aria_project.experiments:
        AriaPeak.AriaPeakXMLPickler.spectra[exp.getName()] = exp.getSpectrum()

    aria_restraints = aria_pickler.load(it_path + '/noe_restraints.xml').peak

    converter = Aria2Ccpn(aria_project, aria_restraints)
    converter.ccpn_project.saveModified()
