"""

          ARIA -- Ambiguous Restraints for Iterative Assignment

                 A software for automated NOE assignment

                               Version 2.2


Copyright (C) Benjamin Bardiaux, Michael Habeck, Therese Malliavin,
              Wolfgang Rieping, and Michael Nilges

All rights reserved.


NO WARRANTY. This software package is provided 'as is' without warranty of
any kind, expressed or implied, including, but not limited to the implied
warranties of merchantability and fitness for a particular purpose or
a warranty of non-infringement.

Distribution of substantively modified versions of this module is
prohibited without the explicit permission of the copyright holders.

$Author: bardiaux $
$Revision: 1.3 $
$Date: 2007/01/09 11:01:57 $
"""


from Settings import Settings
from xmlutils import XMLBasePickler

def decode_ccpn_peak_list_id(x):
    
    exp_name, data_source_name, serial = x.split('|')

    serial = int(serial)

    return exp_name, data_source_name, serial

def decode_ccpn_sequence_id(x):

    mol_system_name, chain_code = x.split('|')

    return mol_system_name, chain_code

## def decode_ccpn_shift_list_id(x):

##     print x

##     shift_list_name, serial = x.split('|')

##     serial = int(serial)

##     return shift_list_name, serial

class CCPNData(Settings):

    def create(self):

        from Settings import MultiTypeEntity, AbsolutePath, TypeEntity
        from TypeChecking import TUPLE

        d = {}
        
        d['filename'] = AbsolutePath(exists=0)

        return d

    def create_default_values(self):
        
        d = {'filename': ''}
        
        return d

## import os, sys

## aria_path = os.environ['ARIA2'] + '/src/py/'

## if not aria_path in sys.path: sys.path.insert(0, aria_path)

## aria imports

from Chain import TYPE_PROTEIN, TYPE_DNA, TYPE_RNA, TYPE_NONPOLYMER

## Conversion settings

CHAIN_TYPE_MAPPING = {'protein': TYPE_PROTEIN,
                      'DNA': TYPE_DNA,
                      'RNA': TYPE_RNA,
                      'nonpolymer': TYPE_NONPOLYMER}

## ccpn imports

CONSTRAINT_LIST_TYPES = ['RdcConstraintList',
                         'DistanceConstraintList',
                         'HBondConstraintList',
                         'JCouplingConstraintList',
                         'DihedralConstraintList']

def makeAriaAppData(ccpn_object):

    from memops.api import Implementation
    from ccpnmr.format.general.Constants import ccpNmr_kw, isAriaInput_kw
    from memops.universal.Constants import True

    Implementation.AppDataBoolean(ccpn_object, application = ccpNmr_kw,
                                  keyword = isAriaInput_kw, value = True)

def setupDialog(dialog, gui):

    from ccpnmr.format.general.userInteraction import setupMultiDialog

    if not dialog:

        if gui is None:

            try:
                import Tkinter
                gui = Tkinter.Tk()
            except:
                gui = None

        dialog = setupMultiDialog(gui)
    
    return dialog, gui

def setupDialogAndSelect(object, dialog, gui, **kw):

    from ccpnmr.format.general.Util import createSelection

    (dialog, gui) = setupDialog(dialog, gui)

    ## now select the object

    (list, dict) = createSelection(object)

    kw['selectionDict'] = dict

    interaction = dialog.SelectionList(gui, list, **kw)

    return interaction, dialog, gui

## BARDIAUX 20060531
def selectConstraintList(ccpn_project, gui, restraint_type = None, msg = None):

    from ccpnmr.format.general.Util import createSelection

    if ccpn_project is None:
        return
    
    structure_generations = ccpn_project.structureGenerations
    nmr_heads = ccpn_project.nmrConstraintHeads

    # first look at structureGeneration if present
    if structure_generations:

        if len(structure_generations) > 1:

            selection_list, selection_dict = createSelection(structure_generations)

            structure_generation = gui_select_structure_generation(selection_list,
                                                                   selection_dict, gui=gui)
        else:
            structure_generation = structure_generations[0]


        ## get ConstraintsList of type restraint_type
        ## via structure_generation's nmrConstraintHead
        if structure_generation:

            nmr_head = structure_generation.nmrConstraintHead

            if nmr_head:

                constraint_lists = nmr_head.\
                                   findAllConstraintLists(className = restraint_type)

                if not constraint_lists:
                    print "No %s Constraints List found." % msg
                    return    

                selection_list, selection_dict = createSelection(constraint_lists)

                constraint_list = gui_select_constraint_list(selection_list, selection_dict, gui=gui)

                if constraint_list is None:
                    return

                key = "%d|%s|%d" % (nmr_head.serial, constraint_list.name, constraint_list.serial)

                print "%s Constraints List %s selected" %(msg, key)

                return key


    ## no generation so we compile all available ConstraintsList for restraint_type
    elif nmr_heads:

        constraint_lists = []
        
        for n in nmr_heads:
            constraint_lists += n.findAllConstraintLists(className = restraint_type)
            
        if not constraint_lists:
            
            print "No %s Constraints List found." % msg
            return    

        selection_list, selection_dict = createSelection(constraint_lists)

        constraint_list = gui_select_constraint_list(selection_list,
                                                     selection_dict, gui=gui)

        if constraint_list is None:
            return

        nmr_head = constraint_list.nmrConstraintHead
        
        key = "%d|%s|%d" % (nmr_head.serial, constraint_list.name, constraint_list.serial)
        
        print "%s Constraints List %s selected" %(msg, key)
            
        return key

    ## no generation and no ConstraintsList
    else:
        
        print "No Constraints List found."

def selectConstraintLists(project):

    from ccpnmr.format.general.Constants import ccpNmr_kw, isAriaInput_kw

    ## select constraint lists if available, and write out for ARIA.
    ## also have to select a structureGeneration for this!!

    constraints = {}

    if not project.structureGenerations: return constraints

    iteration = None

    if len(project.structureGenerations) > 1:

        ## try to fetch tagged structure generation

        for it in project.structureGenerations:
            data = it.findFirstApplicationData(
                application = ccpNmr_kw, keyword = isAriaInput_kw)

            if data and data.value:
                iteration = it
                break

        ## select via user dialog

        if not iteration:

            args = (project.structureGenerations, dialog, gui)
            kw = {'title': 'Select structure generation for ARIA',
                  'text': 'Available structure generations:',
                  'dismissText': 'Cancel'}

            result, dialog, gui = setupDialogAndSelect(*args, **kw)

            if result.isSelected:
                iteration = result.selection
                makeAriaAppData(iteration)

    else:
        iteration = project.structureGenerations[0]

    ## only go on if something selected...

    if iteration:

        head = iteration.nmrConstraintHead

        for type in CONSTRAINT_LIST_TYPES:
            if head.findAllConstraintLists(className = type):
                constraints[type] = []
            else:
                continue 

        ## look if anything tagged available...

        for constraint_list in iteration.nmrConstraintHead.\
                findAllConstraintLists(className = type):
            data = constraint_list.findFirstApplicationData(
                application = ccpNmr_kw, keyword = isAriaInput_kw)

            if data and data.value:
                constraint_lists[type].append(constraint_list)

        ## select constraint lists for output if none found...

        if not constraint_lists[type]:

            ## user interaction

            constraint_lists = structure_generation.nmrConstraintHead.\
                               findAllConstraintLists(className = type)

            args = (list(constraint_lists), dialog, gui)
            kw = {'title': 'Select %s for ARIA' % type,
                  'text': '%s:' % type}

            interaction, dialog, gui = setupDialogAndSelect(*args, **kw)

            if hasattr(interaction,'isSelectedList') and \
                   interaction.isSelectedList:

                constraint_lists[type] = interaction.isSelectedList[:]

                for constraint_list in interaction.isSelectedList:

                    makeAriaAppData(constraint_list)

class Ccpn2Aria:

    def __init__(self, ccpn_molsystem):

        self.ccpn_molsystem = ccpn_molsystem

        ## map ccpn atoms to aria atoms

        self.atom_mapping = {}

        ## create aria molecule and also fills in atom_mapping

        self.aria_molecule = self.makeAriaMolecule()

        ## BARDIAUX for CCPN Constraints
        ## atoms from Aria AtomFactory
        self.aria_atom_from_factory = {}
        ## mapping between Ccpn2Aria atoms and Factory Atoms
        self.atom_to_atom_factory = {}
        
    def saveCcpnProject(self):
  
        self.ccpn_molsystem.root.saveModified()

    def createPeakShiftList(self, ccpn_peak_list, ccpn_shift_list, mol_system_name):
    
        self.atomset_dict = {}
        self.resonance_dict = {}
        self.aria_shift_list = self.getAriaChemicalShiftList(ccpn_shift_list, mol_system_name)
        self.aria_spectrum = self.getAriaSpectrum(ccpn_peak_list)

    def makeAriaMolecule(self, mol_system_code=None):
        """
        Ccpn MolSystem to Aria Molecule conversion code
        """

        from Molecule import Molecule
        
        aria_molecule = Molecule(name = self.ccpn_molsystem.code)

        if mol_system_code is not None:
            chains = self.ccpn_molsystem.chains[mol_system_code]
        else:
            chains = self.ccpn_molsystem.chains

        for ccpn_chain in chains:

            aria_chain = self.makeAriaChain(ccpn_chain)
            aria_molecule.add_chain(aria_chain)

        return aria_molecule

    def makeAriaChain(self, ccpn_chain):

        from Chain import Chain, ChainSettings
        from tools import string_to_segid
        
        aria_settings = ChainSettings()

        ## TODO: below throws KeyError if molType = DNA/RNA

        aria_settings['type'] = CHAIN_TYPE_MAPPING[ccpn_chain.molecule.molType]

        aria_chain = Chain(settings = aria_settings,
                           segid = string_to_segid(ccpn_chain.code))

        for ccpn_residue in ccpn_chain.residues:

            aria_residue = self.makeAriaResidue(ccpn_residue)
            aria_chain.addResidue(aria_residue)
 
        return aria_chain

    def makeAriaResidue(self, ccpn_residue):

        from ccpn2top import setupEquivalentGroup
        from Residue import Residue

        ## TODO: have to define chemComp ARIA sysName for residues
        ## that match, throw error if not recognized (have to define
        ## own topology file for CNS for now)
        
        try:
            aria_residue = Residue(number = ccpn_residue.seqCode,
                                   residue_type = ccpn_residue.ccpCode)

        except:
            message = 'Residue %s not recognized in ARIA - define topology!'
            raise Exception(message %  ccpn_residue.ccpCode)

        ## make atoms

        mapping = {}

        id = 0

        for ccpn_atom in ccpn_residue.atoms:

            aria_atom = self.makeAriaAtom(ccpn_atom, id)
            aria_residue.addAtom(aria_atom)
            mapping[ccpn_atom.name] = aria_atom
            id += 1

        ## set up equivalent groups

        if ccpn_residue.chemCompVar:
            ccpn_chem_comp = ccpn_residue.chemCompVar
        else:
            ccpn_chem_comp = ccpn_residue.molResidue.chemCompLoc

        setupEquivalentGroup(ccpn_chem_comp, mapping, aria_residue)

        return aria_residue

    def makeAriaAtom(self, ccpn_atom, id):
        
        from ccpn2top import setupAriaAtomSettings
        
        ## TODO: use AtomFactory for creating atoms?

        from Atom import Atom, AtomSettings
        from tools import string_to_segid
        
        aria_settings = AtomSettings()
        ccpn_chematom = ccpn_atom.chemAtom

        setupAriaAtomSettings(ccpn_chematom, aria_settings)

        aria_atom = Atom(settings = aria_settings,
                         name = ccpn_atom.name, id=id)

        aria_atom._setSegid(string_to_segid(ccpn_atom.residue.chain.code))

        self.atom_mapping[ccpn_atom] = aria_atom

        return aria_atom

    def getAriaChemicalShift(self, ccpn_object):

        from Datum import ChemicalShift
        
        if ccpn_object is None:
            val = None
            err = None
        else:
            val = ccpn_object.value
            if hasattr(ccpn_object, 'valueError'):
                err = ccpn_object.valueError
            else:
                err = ccpn_object.error

        return ChemicalShift(val, err)
 
    def findCcpnDataDim(self, data_source, exp_dim):

        if exp_dim is None: return None

        data_dim = data_source.findFirstDataDim(expDim = exp_dim)
        if data_dim is None:
            return None
        else:
            return data_dim.dim

    def getAriaChemicalShiftList(self, ccpn_shift_list, mol_system_code):
        """
        Ccpn Experiment to Aria ChemicalShiftList conversion
        """
        from ChemicalShiftList import ChemicalShiftList
        from ShiftAssignment import AVERAGING_METHOD_FAST
        from ShiftAssignment import AVERAGING_METHOD_SLOW
        from ShiftAssignment import AVERAGING_METHOD_NONE
        from ShiftAssignment import ASSIGNMENT_METHOD_STEREO_SPECIFIC
        from ShiftAssignment import ASSIGNMENT_METHOD_EQUIVALENT
        from ShiftAssignment import ASSIGNMENT_METHOD_FLOATING
        from ShiftAssignment import SpinSystem, ShiftAssignment
        from Datum import ChemicalShift

        aria_shift_list = ChemicalShiftList()

        for ccpn_shift in ccpn_shift_list.measurements:

            resonance = ccpn_shift.resonance

            if resonance in self.resonance_dict:
                continue

            resonance_set = resonance.resonanceSet
            
            if not resonance_set:
                continue

            flag = True
            for atom_set in resonance_set.atomSets:
                for atom in atom_set.atoms:
                    if atom.residue.chain.molSystem.code != mol_system_code:
                        flag = False
                        break
                if not flag:
                    break
                
            if not flag:
                continue
                        
            ## this code converts the CCPN assignment status to ARIA

            n_atomsets = len(resonance_set.atomSets)
            n_resonances = len(resonance_set.resonances)

            ## stereospecific assignment

            if n_atomsets == 1 and n_resonances == 1:

                atomset = resonance_set.atomSets[0]

                if atomset in self.atomset_dict:
                    message = 'CCPN atomSet appears more than once'
                    raise Exception(message)

                ccpn_atoms = atomset.atoms

                aria_atoms = [self.atom_mapping[atom] for atom in ccpn_atoms]

                if len(atomset.resonanceSets) <> 1:
                    shiftValues = []
                    for resonanceSet0 in atomset.resonanceSets:
                        for resonance0 in resonanceSet0.resonances:
                            self.resonance_dict[resonance0] = aria_atoms
                            shift0 = resonance0.findFirstShift(parentList=ccpn_shift.parentList)
                            if shift0:
                                shiftValues.append(shift0.value)
                     
                    if len(shiftValues) > 1:

                        residue = atomset.atoms[0].residue
                        res_number = residue.seqCode
                        
                        print 'More than one shift (%s) found for CCPN atom assignment %d.%s. Using arbitrary value %.3f' % \
                              (','.join(['%.3f' % v for v in shiftValues]),res_number, atomset.name,ccpn_shift.value)
                            
                      #message = 'more than one resonance set for CCPN atom set'
                      #raise Exception(message)

                if len(aria_atoms) == 1:
                    method = AVERAGING_METHOD_NONE
                    assignment_type = ASSIGNMENT_METHOD_STEREO_SPECIFIC

                else:
                    method = AVERAGING_METHOD_FAST
                    assignment_type = ASSIGNMENT_METHOD_EQUIVALENT

                aria_spinsystem = SpinSystem(method)
                aria_spinsystem.setAtoms(tuple(aria_atoms))
                aria_shift = self.getAriaChemicalShift(ccpn_shift)
                aria_spinsystem.setChemicalShifts((aria_shift,))
                aria_shiftassignment = ShiftAssignment(assignment_type)
                aria_shiftassignment.setSpinSystems((aria_spinsystem,))

                self.resonance_dict[resonance] = aria_atoms

            ## non-stereospecific assignment

            elif n_atomsets == 2 and n_resonances <= 2:

                for atomset in resonance_set.atomSets:

                    if atomset in self.atomset_dict:
                        message = 'CCPN atomSet appears more than once'
                        raise Exception(message)
                  
                    ccpn_shifts = []
                    ccpn_shift_values = []

                    for resonance in resonance_set.resonances:

                        ccpn_shift = ccpn_shift_list.findFirstMeasurement(\
                            resonance=resonance)
                  
                        if ccpn_shift:

                            value = ccpn_shift.value
                            if value not in ccpn_shift_values:
                                ccpn_shift_values.append(value)
                                ccpn_shifts.append(ccpn_shift)

                            else:
                                pass
##                       print resonance.name
##                       print [x.name for x in resonance_set.atomSets]

                    if len(atomset.resonanceSets) <> 1:
                        otherShiftValues = []
                        for resonanceSet0 in atomset.resonanceSets:
                            for resonance0 in resonanceSet0.resonances:
                                self.resonance_dict[resonance0] = aria_atoms
                                shift0 = resonance0.findFirstShift(parentList=ccpn_shift.parentList)
                                if shift0:
                                    otherShiftValues.append(shift0.value)
 
                        if len(otherShiftValues) != len(ccpn_shift_values):
                            vals1 = ','.join(['%.3f' % v for v in otherShiftValues])
                            vals2 = ','.join(['%.3f' % v for v in ccpn_shift_values])

                            residue = atomset.atoms[0].residue
                            res_number = residue.seqCode
                            
                            print 'More than one shift (%s) found for CCPN atom assignment %d.%s. Using arbitrary values %s' % \
                                  (vals1,res_number,atomset.name,vals2)
 
                            #message = 'more than one resonance set for ' + \
                            #          'CCPN atom set'
                            #raise Exception(message)

                    aria_shifts = []
              
                    for ccpn_shift in ccpn_shifts:

                        aria_shift = self.getAriaChemicalShift(ccpn_shift)
                        aria_shifts.append(aria_shift)

                    if len(aria_shifts) == 1:
                        aria_shifts.append(ChemicalShift(None))

                    aria_spinsystems = []
                    aria_atoms = []

                    for atomset in resonance_set.atomSets:

                        atoms = [self.atom_mapping[a] for a in atomset.atoms]
                        aria_atoms.append(atoms)

                        if len(atoms) == 1:
                            method = AVERAGING_METHOD_NONE
                        else:
                            method = AVERAGING_METHOD_FAST

                        aria_spinsystem = SpinSystem(method)
                        aria_spinsystem.setAtoms(tuple(atoms))
                        aria_spinsystem.setChemicalShifts(tuple(aria_shifts))
                        aria_spinsystems.append(aria_spinsystem)

                    aria_shiftassignment = ShiftAssignment(\
                        ASSIGNMENT_METHOD_FLOATING)
                    aria_shiftassignment.setSpinSystems(tuple(aria_spinsystems))

                    resonances = list(resonance_set.resonances)
                    for ccpn_shift in ccpn_shifts:
                        resonance = ccpn_shift.resonance
                        index = resonances.index(resonance)
                        self.resonance_dict[resonance] = aria_atoms[index]

            else:
                continue

            aria_shift_list.addShiftAssignment(aria_shiftassignment)

        return aria_shift_list

    def makePeakDimAssignments(self, ccpn_peak_dim):

        from Assignment import Assignment, ASSIGNMENT_TYPE_MANUAL

        if ccpn_peak_dim:
            resonances = [x.resonance for x in ccpn_peak_dim.peakDimContribs ]
        else:
            resonances = []

        assignments = []
        for resonance in resonances:
            aria_atoms = self.resonance_dict.get(resonance)
            if not aria_atoms:
                continue
            assignment = Assignment(tuple(aria_atoms), assignment_type =
                                    ASSIGNMENT_TYPE_MANUAL)
            assignments.append(assignment)

        return assignments

    def getAriaSpectrum(self, ccpn_peak_list):
        """
        Ccpn PeakList to Aria (NOESY)Spectrum conversion code 
        """
        from memops.api import Implementation
        from ccpnmr.format.general.Constants import ccpNmr_kw
        from NOESYSpectrum import NOESYSpectrum
        from CrossPeak import CrossPeak
        from Datum import Datum

        data_source = ccpn_peak_list.dataSource

        transfers = data_source.experiment.findAllExpTransfers(
            transferType = 'NOESY')
        
        if (len(transfers) != 1):
            raise Exception('Not a NOESY.')

        light_dim1 = None
        light_dim2 = None
        heavy_dim1 = None
        heavy_dim2 = None

        dim_ref_dict = {}

        for dim_ref in transfers[0].expDimRefs:
            if dim_ref.isotopeCodes != ('1H',):
                raise Exception('not an HH Noesy')

            ## check below is a bit pedantic

            if len(dim_ref.expDim.findAllExpDimRefs(isSplitting = False)) <> 1:
                raise Exception('multiple dimension refs')

            t = dim_ref.findAllExpTransfers(transferType = 'onebond')

            if len(t) == 1:

                t = t[0]            
                if dim_ref == t.expDimRefs[0]:
                    other_dim_ref = t.expDimRefs[1]
                else:
                    other_dim_ref = t.expDimRefs[0]

                if light_dim1 is None:
                    heavy_dim1 = other_dim_ref.expDim
                    light_dim1 = dim_ref.expDim
                    dim_ref_dict['heavy_dim1'] = other_dim_ref
                    dim_ref_dict['light_dim1'] = dim_ref

                else:
                    heavy_dim2 = other_dim_ref.expDim
                    light_dim2 = dim_ref.expDim
                    dim_ref_dict['heavy_dim2'] = other_dim_ref
                    dim_ref_dict['light_dim2'] = dim_ref

            else:

                if light_dim1 is None:
                    heavy_dim1 = None
                    light_dim1 = dim_ref.expDim
                    dim_ref_dict['heavy_dim1'] = None
                    dim_ref_dict['light_dim1'] = dim_ref

                else:
                    heavy_dim2 = None
                    light_dim2 = dim_ref.expDim
                    dim_ref_dict['heavy_dim2'] = None
                    dim_ref_dict['light_dim2'] = dim_ref

        ## tag CCPN dataDimRefs with ARIA dimension mapping
        ## clean up first...
    
        for freq_dim in data_source.dataDims:
            for dim_ref in freq_dim.dataDimRefs:
                data = dim_ref.findFirstApplicationData(
                    application = ccpNmr_kw,
                    keyword = 'ariaDimMapping')

                if data: data.delete()

        for (dim_name, aria_dim) in (('light_dim1', 'Proton1'),
                                     ('light_dim2', 'Proton2'),
                                     ('heavy_dim1', 'Hetero1'),
                                     ('heavy_dim2', 'Hetero2')):
    
            dim_ref = dim_ref_dict[dim_name]
            
            if dim_ref:
                
                exp_dim = dim_ref.expDim
                freq_dim = data_source.findFirstDataDim(expDim = exp_dim)
                data = freq_dim.findFirstDataDimRef(expDimRef = dim_ref)
      
                Implementation.AppDataString(data,
                                             application = ccpNmr_kw,
                                             keyword = 'ariaDimMapping',
                                             value = aria_dim)


        ## continue to set ARIA stuff...
    
        light_dim1 = self.findCcpnDataDim(data_source, light_dim1)
        light_dim2 = self.findCcpnDataDim(data_source, light_dim2)
        heavy_dim1 = self.findCcpnDataDim(data_source, heavy_dim1)
        heavy_dim2 = self.findCcpnDataDim(data_source, heavy_dim2)

        cross_peaks = []

        for ccpn_peak in ccpn_peak_list.peaks:

            ## could have application specific peak numbers stored,
            ## but unless we know which application created the peak
            ## we cannot find this peak number
            ## could pass in (optional) application to create_noesy_spectrum()
            ## and then use that to find peak number
            ## for now use serial

            number = ccpn_peak.serial
            volume = ccpn_peak.findFirstPeakIntensity(
                intensityType = 'volume')

            if volume:
                value = volume.value
                err = volume.error
            else:
                value = None
                err = None

            volume = Datum(value, err)
            
            height = ccpn_peak.findFirstPeakIntensity(intensityType = 'height')
            if height:
                value = height.value
                err = height.error
            else:
                value = None
                err = None

            intensity = Datum(value, err)

            cross_peak = CrossPeak(number = number, volume = volume,
                                   intensity = intensity)

            light_peak_dim1 = ccpn_peak.findFirstPeakDim(dim = light_dim1)
            light_peak_dim2 = ccpn_peak.findFirstPeakDim(dim = light_dim2)
            heavy_peak_dim1 = ccpn_peak.findFirstPeakDim(dim = heavy_dim1)
            heavy_peak_dim2 = ccpn_peak.findFirstPeakDim(dim = heavy_dim2)
 
            cross_peak.setProton1ChemicalShift(
                self.getAriaChemicalShift(light_peak_dim1))
            cross_peak.setProton2ChemicalShift(
                self.getAriaChemicalShift(light_peak_dim2))
            cross_peak.setHetero1ChemicalShift(
                self.getAriaChemicalShift(heavy_peak_dim1))
            cross_peak.setHetero2ChemicalShift(
                self.getAriaChemicalShift(heavy_peak_dim2))

            light_assignments1 = self.makePeakDimAssignments(light_peak_dim1)
            light_assignments2 = self.makePeakDimAssignments(light_peak_dim2)
            heavy_assignments1 = self.makePeakDimAssignments(heavy_peak_dim1)
            heavy_assignments2 = self.makePeakDimAssignments(heavy_peak_dim2)

            [cross_peak.addProton1Assignment(a) for a in light_assignments1]
            [cross_peak.addProton2Assignment(a) for a in light_assignments2]
            [cross_peak.addHetero1Assignment(a) for a in heavy_assignments1]
            [cross_peak.addHetero2Assignment(a) for a in heavy_assignments2]

            cross_peaks.append(cross_peak)
 
        spectrum = NOESYSpectrum(name = data_source.experiment.name,
                                 noes = cross_peaks)

        return spectrum

    ## BARDIAUX DistanceRestraints
    def getAriaAtomsFromFixedResonance(self, fixedResonance):

        atoms = []

        if self.ccpn_resonance_to_atoms.has_key(fixedResonance):
            res_to_atoms = self.ccpn_resonance_to_atoms[fixedResonance]
        else:
            return None
        
        if not res_to_atoms : print fixedResonance
        for r in res_to_atoms:
            chain = self.ccpn_molsystem.findFirstChain(code = r.chain.code)
            resid = chain.findFirstResidue(seqId = r.getResidue().seqId)
            if r.chemAtomSet:
                chem_atoms = r.chemAtomSet.chemAtoms
            else:
                chem_atoms = [r.chemAtom]

            for ca in chem_atoms:
                ccpn_atom = resid.findFirstAtom(chemAtom = ca)
                if ccpn_atom:
                    aria_atom = self.atom_mapping[ccpn_atom]
                    ## Here we nedd to find aria Atom from the factory
                    if self.atom_to_atom_factory.has_key(aria_atom):
                        atom = self.atom_to_atom_factory[aria_atom]
                    else:    
                        atom = [a for a in self.aria_atom_from_factory if a.getSegid() == aria_atom.getSegid() and \
                                a.getResidue().getNumber() == aria_atom.getResidue().getNumber() and \
                                a.getName() == aria_atom.getName()][0]
                        
                        self.atom_to_atom_factory[aria_atom] = atom
                    
                    atoms.append(atom)
                else:
                    print 'No aria atoms found for %s, %s' % (r.getAtomId(), ca)
                    
        return atoms
    
    ## BARDIAUX DistanceRestraints    
    #def getAriaDistanceRestraintsList(self, constraint_list, aria_mol):
    def getAriaDistanceRestraintsList(self, constraint_list, constraint_type, aria_mol):    
        """
        Convert CCPN DistanceContraintList
        to list of ARIA DistanceRestraint
        """
        
        from ShiftAssignment import SpinSystem, AVERAGING_METHOD_NONE
        import Contribution as C
        from Singleton import SpinPairFactory
        from AriaPeak import DistanceRestraint
        from ccpnmr.format.converters.CnsFormat import CnsFormat
        import DataContainer as DC
                
        ## A bit tricky :
        ## to map CCPN FixedResonance of DistanceConstraintItems
        ## with ARIA atoms, we use the CnsFormat from formatConverter.
        ## a simpler way to that may exist but I did'nt find it yet....
        ccpn_project = constraint_list.root #nmrConstraintHead.project

        cns_format = CnsFormat(ccpn_project)

        f = {DC.DATA_UNAMBIGUOUS : cns_format.writeDistanceConstraints,
             DC.DATA_AMBIGUOUS   : cns_format.writeDistanceConstraints,
             DC.DATA_HBONDS      : cns_format.writeHBondConstraints}

        kw = {'compressResonances' : 0,
              'minimalPrompts' : 1,
              'noWrite' : 1,
              'forceNamingSystem' : 'IUPAC'}

        f[constraint_type]('', constraintList = constraint_list, **kw)

        self.ccpn_resonance_to_atoms = cns_format.resonanceToAtoms
        
        if not self.aria_atom_from_factory:
            self.aria_atom_from_factory = [a for c in aria_mol.get_chains()\
                                           for r in c.getResidues()\
                                           for a in r.getAtoms()]
        
        SpinPair = SpinPairFactory()

        restraints = []

##         source =  "%d|%s|%d" % (constraint_list.nmrConstraintHead.serial,
##                                 constraint_list.name,
##                                 constraint_list.serial)
        
        source =  "%d_%s_%d.tbl" % (constraintList.nmrConstraintStore.serial,
                                    constraintList.name,
                                    constraintList.serial)
        
        msg = "WARNING : %s ignored."
        
        for distConstr in constraint_list.constraints:

            contributions = []

            restraint = DistanceRestraint()
            
            target, upper, lower = distConstr.targetValue, distConstr.upperLimit, distConstr.lowerLimit
            weight = distConstr.weight

            restraint.setDistance(target)
            restraint.setLowerBound(lower)
            restraint.setUpperBound(upper)
            restraint.setWeight(weight)

            restraint.setNumber(int(distConstr.serial))
            restraint.setSource(source)
            
            for constrItem in distConstr.items:

                reso1, reso2 = constrItem.resonances

                ss1 = SpinSystem(AVERAGING_METHOD_NONE)
                atoms = self.getAriaAtomsFromFixedResonance(reso1)

                if not atoms:
                    print msg % (constrItem)
                    continue
                
                ss1.setAtoms(tuple(atoms))
                    

                ss2 = SpinSystem(AVERAGING_METHOD_NONE)
                atoms = self.getAriaAtomsFromFixedResonance(reso2)
                
                if not atoms:
                    print msg % (constrItem)
                    continue
                
                ss2.setAtoms(tuple(atoms))
                
                spin_pairs = []
                for a1 in ss1.getAtoms():
                    for a2 in ss2.getAtoms():

                        if a1 == a2:
                            continue

                        sp = SpinPair(a1, a2)
                        spin_pairs.append(sp)

                if not len(spin_pairs):
                    continue

                c = C.Contribution(0,
                                   C.CONTRIBUTION_TYPE_FAST_EXCHANGE,
                                   spin_pairs, spin_systems = (ss1, ss2))



                ## TODO: hack
                c.setSpinSystems((ss1, ss2))

                contributions.append(c)

                ## Need a weight if no further Contribution evaluator
                c.setWeight(1.)

            restraint.setContributions(contributions)

            restraints.append(restraint)

        return restraints


    ## BARDIAUX Restraints
    def dumpRestraintsList(self, constraint_list, data_dir, type):
        
        """
        Dump CCPN RestraintList to CNS Format TBL file.
        """
        
        import DataContainer as DC
        import os
        from ccpnmr.format.converters.CnsFormat import CnsFormat

        ccpn_project = constraint_list.nmrConstraintHead.project

        cns_format = CnsFormat(ccpn_project)

        f = {DC.DATA_UNAMBIGUOUS : cns_format.writeDistanceConstraints,
             DC.DATA_AMBIGUOUS   : cns_format.writeDistanceConstraints,
             DC.DATA_HBONDS      : cns_format.writeHBondConstraints,
             DC.DATA_DIHEDRALS   : cns_format.writeDihedralConstraints,
             DC.DATA_RDCS        : cns_format.writeRdcConstraints,
             DC.DATA_KARPLUS     : cns_format.writeJCouplingConstraints,
             DC.DATA_SSBONDS     : cns_format.writeDistanceConstraints}
             
        
        kw = {'compressResonances' : 0,
              'minimalPrompts' : 1,
              'forceNamingSystem' : 'XPLOR'}


        file_name =  "%d_%s_%d.tbl" % (constraint_list.nmrConstraintHead.serial,
                                       constraint_list.name.replace(" ", ""),
                                       constraint_list.serial)
        
        dst = os.path.join(data_dir, file_name)
        
        # assumes that Resonnances are already linked
        # overwrite existing file without warning
        
        f[type](dst, constraintList = constraint_list, **kw)

        return dst


def gui_select_molsystem(ccpn_project, dialog=None, gui=None):
    """
    gui: tk parent widget
    """

    args = (ccpn_project.molSystems, dialog, gui)
    
    kw = {'title': 'Select molecular system',
          'text': 'Existing molecular system codes:',
          'urlFile': 'SelectMolSystem'}

    interaction, dialog, gui = setupDialogAndSelect(*args, **kw)

    if interaction.isSelected:
        molsystem = interaction.selection
    else:
        molsystem = None
        
    if molsystem:
            
        ## TODO: move to else: below (after if not molsystem),
        ## but then check if already tagged!

        makeAriaAppData(molsystem)

    return molsystem

def gui_select_shift_list(selection_list, selection_dict, dialog=None,
                          gui=None):
                    
    dialog, gui = setupDialog(dialog, gui)

    select_list = dialog.SelectionList(gui, selection_list,
                                       selectionDict = selection_dict,
                                       title = 'Select shift list for this peak list',
                                       text = 'Valid shift lists:',
                                       dismissText = 'No valid shift list')

    if select_list.isSelected:
        shift_list = select_list.selection
    else:
        shift_list = None
        
    return shift_list

def gui_select_peak_list(peaklist_labels, peaklist_dict, dialog=None,
                         gui=None):

    dialog, gui = setupDialog(dialog, gui)

    select = dialog.SelectionList(gui, peaklist_labels,
                                  selectionDict = peaklist_dict,
                                  title = 'Select peak list for ARIA',
                                  text = 'Remaining valid peak lists:',
                                  dismissText = 'Finish selection')

    if select.isSelected:
        peak_list = select.selection
##         makeAriaAppData(peak_list)

    else:
        peak_list = None

    return peak_list

## BARDIAUX CCPN restraints
def gui_select_constraint_list(constraintlist_labels, constraintlist_dict, dialog=None,
                               gui=None):

    dialog, gui = setupDialog(dialog, gui)

    select = dialog.SelectionList(gui, constraintlist_labels,
                                  selectionDict = constraintlist_dict,
                                  title = 'Select Constraint list for ARIA',
                                  text = 'Remaining valid constraint lists:',
                                  dismissText = 'Finish selection')


    if select.isSelected:
        constraint_list = select.selection
##         makeAriaAppData(peak_list)

    else:
        constraint_list = None

    return constraint_list

def gui_select_structure_generation(generation_labels, generation_dict, dialog=None,
                                    gui=None):

    dialog, gui = setupDialog(dialog, gui)

    select = dialog.SelectionList(gui, generation_labels,
                                  selectionDict = generation_dict,
                                  title = 'Select Structure Generation for ARIA',
                                  text = 'Remaining valid structure generation:',
                                  dismissText = 'Finish selection')

    if select.isSelected:
        structure_generation  = select.selection
##         makeAriaAppData(peak_list)

    else:
        structure_generation = None

    return structure_generation

def gui_select_nmr_constraint_head(nmrconstrainthead_labels, nmrconstrainthead_dict, dialog=None,
                                    gui=None):

    dialog, gui = setupDialog(dialog, gui)

    select = dialog.SelectionList(gui, nmrconstrainthead_labels,
                                  selectionDict = nmrconstrainthead_dict,
                                  title = 'Select Structure Generation for ARIA',
                                  text = 'Remaining valid structure generation:',
                                  dismissText = 'Finish selection')

    if select.isSelected:
        nmrconstrainthead  = select.selection
##         makeAriaAppData(peak_list)

    else:
        nmrconstrainthead = None

    return nmrconstrainthead

##

def get_shift_and_peak_lists(ccpn_project, molsystem_id):

    ## get relevant peak and shift lists
    ## slow because have to loop over resonances...

    molsystem_name, chain_code = decode_ccpn_sequence_id(molsystem_id)

    peak_lists = []

    if len(ccpn_project.molSystems) > 1:

        shift_lists = []

        resonances = []

        molsystem = [x for x in ccpn_project.molSystems \
                     if str(x.code) == molsystem_code][0]

        chain = [c for c in molsystem.chains if str(c.code) == chain_code]

        if not chain:
            raise ValueError('Could not find chain "%s" in molecular system "%s".' % \
                             (chain_code, mol_system_name))

        chain = chain[0]
        
        for residue in chain.residues:
            for atom in residue.atoms:
                if atom.atomSet:
                    for res_set in atom.atomSet.resonanceSets:
                        for resonance in res_set.resonances:
                            if resonance not in resonances:
                                resonances.append(resonance)

        for resonance in resonances:
            for contrib in resonance.peakDimContribs:
                peak_list = contrib.peakDim.peak.peakList
                if peak_list not in peak_lists:
                    peak_lists.append(peak_list)

        for resonance in resonances:
            for shift in resonance.shifts:
                if shift.parentList not in shift_lists:
                    shift_lists.append(shift.parentList)

    else:

        shift_lists = list(ccpn_project.findAllNmrMeasurementLists(\
            className = 'ShiftList'))

        for experiment in ccpn_project.nmrExperiments:
            for data_source in experiment.dataSources:
                for peak_list in data_source.peakLists:
                    peak_lists.append(peak_list)

    return shift_lists, peak_lists

def find_shift_and_peak_list(ccpn_project, molsystem_id, peak_list_key,
                             shift_list_name):

    ## where experiment_name is Experiment.name
    ##       shift_list_name is ShiftList.name
    ##       peak_list_name is PeakList.name

##    shift_list_name, shift_list_serial = decode_ccpn_shift_list_id(shift_list_id)

    peak_list_exp_name, peak_list_name, peak_list_serial = \
                        decode_ccpn_peak_list_id(peak_list_key)

    molsystem_code, chain_code = molsystem_id

    ## get relevant peak and shift lists
    ## slow because have to loop over resonances...

    peak_lists = []

    if len(ccpn_project.molSystems) > 1:

        shift_lists = []

        resonances = []

        molsystem = [x for x in ccpn_project.molSystems \
                     if x.code == molsystem_code][0]

        chain = [c for c in molsystem.chains if str(c.code) == chain_code]

        if not chain:
            raise ValueError('Could not find chain "%s" in molecular system "%s".' % \
                             (chain_code, molsystem_code))

        chain = chain[0]
        
        for residue in chain.residues:
            for atom in residue.atoms:
                if atom.atomSet:
                    for res_set in atom.atomSet.resonanceSets:
                        for resonance in res_set.resonances:
                            if resonance not in resonances:
                                resonances.append(resonance)

        for resonance in resonances:
            for contrib in resonance.peakDimContribs:
                peak_list = contrib.peakDim.peak.peakList
                if peak_list not in peak_lists:
                    peak_lists.append(peak_list)

        for resonance in resonances:
            for shift in resonance.shifts:
                if shift.parentList not in shift_lists:
                    shift_lists.append(shift.parentList)

    else:

        shift_lists = list(ccpn_project.findAllNmrMeasurementLists(\
            className = 'ShiftList'))

        for experiment in ccpn_project.nmrExperiments:
            for data_source in experiment.dataSources:
                for peak_list in data_source.peakLists:
                    peak_lists.append(peak_list)

    sl = [x for x in shift_lists if str(x.name) == shift_list_name]

    if len(sl) <> 1:
        print 'ERROR: key "%s" for shift list is ambiguous' % shift_list_name
        sl = []

    peak_lists = [x for x in peak_lists if \
                  str(x.dataSource.name) == peak_list_name and \
                  x.serial == peak_list_serial]

    ## find peak_list

    pl = [x for x in peak_lists if \
          x.dataSource.parent.name == peak_list_exp_name]

    if len(pl) > 1:
        print 'ERROR: key "%s" for peak list is ambiguous.' % peak_list_key

        pl = []

    return sl, pl

def ariaSetupFromCcpn(ccpn_project, molsystem_id=None,
                      experiment_names=None):

    """
    experiment_names: tuple of 2-tuples:((peak_list_name, shift_list_name),..).
    """

    from ccpnmr.format.general.Constants import ccpNmr_kw, isAriaInput_kw
    from ccpnmr.format.general.Util import createSelection

    dialog = None
    gui = None

    molsystem = None

    ## what is that?

##     if mol_system_name is not None:

##         molsystem = [x for x in ccpn_project.molSystems \
##                      if x.code == mol_system_name]

##         if molsystem:
##             molsystem = molsystem[0]
            
##         else:

##             print 'Molecular system "%s" not found.' % mol_system_name
##             raise NameEror

    if molsystem is None:

        ## Set up molSystem - see if tagged one available

        for mol in ccpn_project.molSystems:

            data = mol.findFirstApplicationData(application = ccpNmr_kw,
                                                keyword = isAriaInput_kw)

            if data and data.value:
                molsystem = mol
                break

        ## if no tagged one, select a molSystem
            
        if not molsystem and ccpn_project.molSystems:
            
            if len(ccpn_project.molSystems) > 1:
                molsystem = gui_select_molsystem(ccpn_project, dialog, gui)
                
            else:
                molsystem = ccpn_project.molSystems[0]


    if not molsystem:
        raise ValueError("No CCPN molSystem present or selected!")

    ## now set up the which peak/shift lists are relevant for
    ## this molSystem...
    shift_lists, peak_lists = get_shift_and_peak_lists(ccpn_project,
                                                       molsystem_id)

    print shift_lists, peak_lists
                                                       
    if experiment_names:

        names_dict = {}

        for peak_list_name, shift_list_name in experiment_names:
            names_dict[peak_list_name] = shift_list_name
        
        shift_lists, peak_lists = get_shift_and_peak_lists(ccpn_project,
                                                           molsystem_id)
            
        if not peak_lists or not shift_lists:

            print "ERROR: no CCPN peak or shift lists present for" + \
                  "molecular system! Aborting..."
            
            return None

        if names_dict:

            selected_peaklists = [pl for pl in peak_lists if \
                                  pl.dataSource.name in names_dict.keys()]

            print 'selected_peaklists', selected_peaklists
            print [x.dataSource.name for x in selected_peaklists]

            if not selected_peaklists:

                print 'None of the peak lists "%s" found.' % \
                      str(names_dict.keys())

                raise NameError

        else:

            ## determine if any peak lists are tagged

            selected_peaklists = []

            for peak_list in peak_lists:

                data = peak_list.findFirstApplicationData(application = ccpNmr_kw, keyword = isAriaInput_kw)

                if data and data.value:
                    selected_peaklists.append(peak_list)

        ## set up peakList / shiftList pairs

        ## loop where either the preselected peaklists are handled or
        ## peaklists for aria are selected... the relevant shift list
        ## is then automatically or manually selected.

        peak_shift_lists = []

        (peak_list_labels, peak_list_dict) = createSelection(peak_lists)

        select = 1

        while select:

            peak_list = None

            ## if lists available, select from there... else user interaction

            if selected_peaklists:
                peak_list = selected_peaklists.pop()

                if not selected_peaklists: select = None

            else:

                peak_list = gui_select_peak_list(peak_list_labels,
                                                 peak_list_dict,
                                                 dialog, gui)

        ## if there is a peaklist, see if can find the shift list... 

            if peak_list:

                shift_list_name = names_dict[peak_list.dataSource.name]

                shift_list = [x for x in shift_lists if x.name == shift_list_name]

                if shift_list:
                    shift_list = shift_list[0]

                else:
                    print 'Shift list "%s" not found.' % shift_list_name

                    raise NameError

    ##                shift_list = peak_list.dataSource.experiment.shiftList

                ## never None ...
                ## TODO: graphical selection still needed?

                if not shift_list:

                    selection_list,selection_dict=createSelection(shift_lists)


                    shift_list = gui_select_shift_list(selection_list,
                                                       selection_dict,
                                                       dialog, gui)


                if shift_list:

                    peak_shift_lists.append((peak_list, shift_list))

                    for label in peak_list_labels:

                        if peak_list_dict[label] == peak_list:
                            del peak_list_dict[label]
                            peak_list_labels.pop(peak_list_labels.index(label))
                            break

                else:

                    for label in peak_listlabels:
                        if peak_list_dict[label] == peak_list:
                            message = "  Warning: no shift list selected " + \
                                      "for peak list %s" 
                            print message % label

            else:

                select = None

    else:
        peak_shift_lists = []

    ## select RDC lists if available, and write out for ARIA.
    ## also have to select a structureGeneration for this!!

    constraint_lists = {}

    if ccpn_project.structureGenerations:

        structure_generation = None

        if len(ccpn_project.structureGenerations) > 1:

            for generation in ccpn_project.structureGenerations:
                data = generation.findFirstApplicationData(\
                    application = ccpNmr_kw, keyword = isAriaInput_kw)

                if data and data.value:
                    structure_generation = generation
                    break

            if not structure_generation:

                args = (ccpn_project.structureGenerations, dialog, gui)
                kw = {'title': 'Select structure generation for ARIA',
                      'text': 'Available structure generations:',
                      'dismissText': 'Cancel'}

                interaction, dialog, gui = setupDialogAndSelect(*args, **kw)

                if interaction.isSelected:

                    structure_generation = interaction.selection
                    makeAriaAppData(structure_generation)
                    
        else:

          structure_generation = ccpn_project.structureGenerations[0]

        ## only go on if something selected...

        if structure_generation:

          for type in CONSTRAINT_LIST_TYPES:

            if structure_generation.nmrConstraintHead.findAllConstraintLists(
                className = type):
                constraint_lists[type] = []
            else:
              continue 

            ## look if anything tagged available...

            for constraint_list in structure_generation.nmrConstraintHead.\
                    findAllConstraintLists(className = type):
                data = constraint_list.findFirstApplicationData(
                    application = ccpNmr_kw, keyword = isAriaInput_kw)

                if data and data.value:
                    constraint_lists[type].append(constraint_list)

            ## select constraint lists for output if none found...

            if not constraint_lists[type]:

                ## user interaction

                constraint_lists = structure_generation.nmrConstraintHead.\
                                   findAllConstraintLists(className = type)

                args = (list(constraint_lists), dialog, gui)
                kw = {'title': 'Select %s for ARIA' % type,
                      'text': '%s:' % type}

                interaction, dialog, gui = setupDialogAndSelect(*args, **kw)

                if hasattr(interaction,'isSelectedList') and \
                       interaction.isSelectedList:

                    constraint_lists[type] = interaction.isSelectedList[:]

                    for constraint_list in interaction.isSelectedList:

                        makeAriaAppData(constraint_list)
                        
    ## TODO: write out RDC list here?!?
                    
    return molsystem, peak_shift_lists, constraint_lists

## newer version w/o gui stuff 

def ariaSetupFromCcpn2(ccpn_project, molsystem_id, exp_names, restraints_names):

    """
    experiment_names: tuple of 2-tuples:((peak_list_name, shift_list_name),..).
    
    restraints_names : dict with key = Aria DC DataTypes and value = list of ccpn_ids
    """

    from ccpnmr.format.general.Constants import ccpNmr_kw, isAriaInput_kw
    from ccpnmr.format.general.Util import createSelection

    molsystem = None

    molsystem_name, chain_code = molsystem_id

    if molsystem_name is not None:

        molsystem = [x for x in ccpn_project.molSystems \
                     if x.code == molsystem_name]

        if molsystem:
            molsystem = molsystem[0]

        else:

            print 'Molecular system "%s" not found.' % molsystem_name
            raise NameEror

    if molsystem is None:

        ## Set up molSystem - see if tagged one available

        for mol in ccpn_project.molSystems:

            data = mol.findFirstApplicationData(application = ccpNmr_kw,
                                                keyword = isAriaInput_kw)

            if data and data.value:
                molsystem = mol
                break

        ## if no tagged one, select a molSystem
            
        if not molsystem and ccpn_project.molSystems:
            
            if len(ccpn_project.molSystems) > 1:
                molsystem = gui_select_molsystem(ccpn_project, dialog, gui)
                
            else:
                molsystem = ccpn_project.molSystems[0]

    if not molsystem:
        raise ValueError("No CCPN molSystem present or selected!")

    chain = [c for c in molsystem.chains if str(c.code) == chain_code]

    if not chain:
        raise ValueError('Could not find chain "%s" in molecular system "%s".' % \
        (chain_code, molsystem_name))

    ## now set up the peak/shift lists which are relevant for
    ## this molSystem...

    peak_shift_lists = []

    for peak_list_key, shift_list_name in exp_names:

        shift_list, peak_list = find_shift_and_peak_list(ccpn_project,
                                                         molsystem_id,
                                                         peak_list_key,
                                                         shift_list_name)

        if shift_list and peak_list:
            peak_shift_lists.append((peak_list[0], shift_list[0]))

    if not peak_shift_lists:

        # BARDIAUX 2.2
        # We want to use CCPN data but only restraints (no Spectrum)
        print "No CCPN peak or shift lists present for " + \
              "molecular system!"
        
##         print "ERROR: no CCPN peak or shift lists present for " + \
##               "molecular system! Aborting..."

##         return None

    ## select RDC lists if available, and write out for ARIA.
    ## also have to select a structureGeneration for this!!

    constraint_lists = {}
    
    ## BARDIAUX : retrieve Constraint List from serials/names
    for type, names_list in restraints_names.items():
        
        constraint_lists[type] = []
        
        for name in names_list:

            head_serial, constraint_list_name, constraint_list_serial = name.split('|')
            head_serial = int(head_serial)
            constraint_list_serial = int(constraint_list_serial)

            head = ccpn_project.findFirstNmrConstraintHead(serial = head_serial)

            constraint_list = head.findFirstConstraintList(name = constraint_list_name,
                                                          serial = constraint_list_serial)
            if constraint_list:
                constraint_lists[type].append(constraint_list)
            else:

                print "ERROR: Nmr Constraint List %s cannot be found in CCPN project %s" %\
                      (constraint_list_name, ccpn_project.name)
    

##     if 0 and ccpn_project.structureGenerations:

##         structure_generation = None

##         if len(ccpn_project.structureGenerations) > 1:

##             for generation in ccpn_project.structureGenerations:
##                 data = generation.findFirstApplicationData(\
##                     application = ccpNmr_kw, keyword = isAriaInput_kw)

##                 if data and data.value:
##                     structure_generation = generation
##                     break

##             if not structure_generation:

##                 args = (ccpn_project.structureGenerations, dialog, gui)
##                 kw = {'title': 'Select structure generation for ARIA',
##                       'text': 'Available structure generations:',
##                       'dismissText': 'Cancel'}

##                 interaction, dialog, gui = setupDialogAndSelect(*args, **kw)

##                 if interaction.isSelected:

##                     structure_generation = interaction.selection
##                     makeAriaAppData(structure_generation)
                    
##         else:

##           structure_generation = ccpn_project.structureGenerations[0]

##         ## only go on if something selected...

##         if structure_generation:

##           for type in CONSTRAINT_LIST_TYPES:

##             if structure_generation.nmrConstraintHead.findAllConstraintLists(
##                 className = type):
##                 constraint_lists[type] = []
##             else:
##               continue 

##             ## look if anything tagged available...

##             for constraint_list in structure_generation.nmrConstraintHead.\
##                     findAllConstraintLists(className = type):
##                 data = constraint_list.findFirstApplicationData(
##                     application = ccpNmr_kw, keyword = isAriaInput_kw)

##                 if data and data.value:
##                     constraint_lists[type].append(constraint_list)

##             ## select constraint lists for output if none found...

##             if not constraint_lists[type]:

##                 ## user interaction

##                 constraint_lists = structure_generation.nmrConstraintHead.\
##                                    findAllConstraintLists(className = type)

##                 args = (list(constraint_lists), dialog, gui)
                
##                 kw = {'title': 'Select %s for ARIA' % type,
##                       'text': '%s:' % type}

##                 interaction, dialog, gui = setupDialogAndSelect(*args, **kw)

##                 if hasattr(interaction,'isSelectedList') and \
##                        interaction.isSelectedList:

##                     constraint_lists[type] = interaction.isSelectedList[:]

##                     for constraint_list in interaction.isSelectedList:

##                         makeAriaAppData(constraint_list)
                        
##     ## TODO: write out RDC list here?!?
                    
    return molsystem, peak_shift_lists, constraint_lists

class CCPNDataXMLPickler(XMLBasePickler):

    order = 'filename',

    def create(self):
        return CCPNData()

    def _xml_state(self, x):

        from xmlutils import XMLElement

        e = XMLElement()

        e.filename = x['filename']

        order = list(self.order)

        e.set_tag_order(order)

        return e

    def load_from_element(self, e):

        from tools import as_tuple
        
        s = self.create()

        filename = str(e.filename).strip()

        E = s.getEntity('filename')

        E.reset()
        E.mandatory(filename <> '')

        s['filename'] = filename

        return s

CCPNData._xml_state = CCPNDataXMLPickler()._xml_state

